/** 
 * Copyright (c) IBM Corporation, 2006
 * This software and documentation is the confidential and proprietary
 * information of IBM, Corp. ("Confidential Information").
 * Raleigh, NC USA
 */

package com.ibm.jpos.tools.sdicc.demo;

import java.awt.Dimension;
import java.awt.Toolkit;

import javax.swing.JFrame;

import jpos.JposConst;
import jpos.JposException;
import jpos.FiscalPrinter;
import jpos.events.StatusUpdateEvent;
import jpos.events.StatusUpdateListener;
import jpos.events.DirectIOEvent;
import jpos.events.DirectIOListener;
import com.ibm.jpos.services.DirectIO;

/**
 * FiscalPrinterDemo class implements a demo for the basic functionality of the
 * FiscalPrinter
 * 
 * @author
 */

public class FiscalPrinterDemo implements DeviceDemo, StatusUpdateListener,
        DirectIOListener
{

    // -------------------------------------------------------------------------
    // public methods
    //

    /**
     * main method creates a GUI to try FiscalPrinterDemo
     */

    public static void main(String args[])
    {
        // main needs one logical name as argument
        if (args.length != 1)
        {
            System.out.println("Usage : \n"
                             + "java com.ibm.jpos.tools.sdicc.demo.FiscalPrinterDemo <logicalName>");
            return;
        }
        // create this class that encapsulate the jpos.FiscalPrinter
        // functionality
        DeviceDemo devDemo = new FiscalPrinterDemo();
        // create the UI for this demo
        DemoUI demoUI = new DemoUI(args, devDemo);
        // Add the ui to a JFrame
        JFrame mainWindow = new JFrame("FiscalPrinterDemo");
        mainWindow.setContentPane(demoUI);
        mainWindow.setSize(700, 400);
        // Center the Frame
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();

        mainWindow.setLocation((screenSize.width / 2)
                - (mainWindow.getSize().width / 2), (screenSize.height / 2)
                - (mainWindow.getSize().height / 2));

        mainWindow.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        mainWindow.setVisible(true);
    }

    /**
     * @return a bidimensional arrays of strings the first array has the button
     *         labels the seccond array has the tooltips for those buttons
     */
    public String[][] getButtonLabels()
    {
        return new String[][]
        {
        { TEXT_FPTR_TEST },
        { TOOLTIP_TEXT_FPTR_TEST } };

    }

    /**
     * @return the devCat for this device
     */
    public String getDevCat()
    {
        return "FiscalPrinter";
    }

    /**
     * @param demoUI
     *            the ui to be used for this demo
     */
    public void setUI(DemoUI demoUI)
    {
        ui = demoUI;
    }

    /**
     * @param button
     *            the button that was pressed
     */
    public void buttonPressed(String button) throws JposException
    {
        if (button.equals(TEXT_FPTR_TEST))
        {
            // start the device test
            doStartDeviceTest();
        }

    }

    /**
     * implementation of statusUpdateListener interface it is called when the
     * status of the device is updated
     * 
     * @param e
     *            the event generated by the device
     */
    public void statusUpdateOccurred(StatusUpdateEvent e)
    {
        int status = e.getStatus();

        String statusMsg = "StatusUpdateOccurred = ";
        switch (status)
        {
            case JposConst.JPOS_SUE_POWER_ONLINE:
                statusMsg += "JPOS_SUE_POWER_ONLINE";
                break;

            case JposConst.JPOS_SUE_POWER_OFF_OFFLINE:
            case JposConst.JPOS_SUE_POWER_OFFLINE:
                statusMsg += "JPOS_SUE_POWER_OFF_OFFLINE";
                break;

        }
        // display the status update received
        ui.displayText(statusMsg);

    }

    /**
     * implementation of directIOListener interface it is called when the the
     * device generates a DirectIO event
     * 
     * @param e
     *            the event generated by the device
     */
    public void directIOOccurred(DirectIOEvent arg0)
    {
        String statusMsg = null;
        String dataMsg = null;
        String objMsg = null;

        switch (arg0.getEventNumber())
        {

            // here, depending on the event we show the incoming parameters
            case DirectIO.FISCAL_STATUS:
                statusMsg = new String("FISCAL_STATUS");
                dataMsg = Integer.toHexString(arg0.getData());
                objMsg = "Last Cmd: "
                        + Integer.toHexString(((Integer) (arg0.getObject()))
                                .intValue());
                synchronized (statusReceived)
                {
                    statusReceived.notify();
                }
                break;
            case DirectIO.FISCAL_ERROR:
                statusMsg = new String("FISCAL_ERROR");
                dataMsg = "FP RC: " + Integer.toHexString(arg0.getData());
                objMsg = "mp2: " + (((Integer) (arg0.getObject())).intValue());
                break;
            case DirectIO.FISCAL_DATA_AVAIL:
                statusMsg = new String("FISCAL_DATA_AVAIL");
                dataMsg = Integer.toHexString(arg0.getData());
                objMsg = "Data length: "
                        + Integer.toHexString(((Integer) (arg0.getObject()))
                                .intValue());
                break;
            case DirectIO.FISCAL_RAW_STATUS:
                statusMsg = new String("FISCAL_RAW_STATUS");
                dataMsg = Integer.toHexString(arg0.getData());
                objMsg = "Data: "
                        + byteArrayToHexString((byte[]) arg0.getObject());
                break;
            case DirectIO.FISCAL_IPL_END_STATUS:
                statusMsg = new String("FISCAL_IPL_END_STATUS");
                dataMsg = Integer.toHexString(arg0.getData());
                objMsg = "Data: "
                        + byteArrayToHexString((byte[]) arg0.getObject());
                synchronized (iplEndReceived)
                {
                    iplEndReceived.notify();
                }

                break;
            default:
                statusMsg = Integer.toHexString(arg0.getData());
                dataMsg = Integer.toHexString(arg0.getData());
                objMsg = Integer.toHexString(((Integer) (arg0.getObject()))
                        .intValue());
                break;
        }
        // shows the values
        ui.displayText("> DIRECTIO sts: " + statusMsg + " data: " + dataMsg
                + " obj:" + objMsg);

        // here we do post-processing. If we have a FISCAL_DATA_AVAIL,
        // the we read the available data from the fiscal printer.
        switch (arg0.getEventNumber())
        {
            case DirectIO.FISCAL_DATA_AVAIL:
                read(((Integer) (arg0.getObject())).intValue());
                break;
        }
    }

    // -------------------------------------------------------------------------
    // UPOS Functionality
    //

    // -------------------------------------------------------------------------
    // Protected Methods
    //

    /**
     * Open claim and enable the device, display some properties,
     * executes some cmds and wait for user generated events.   
     * Then close the device.
     */
    protected void doStartDeviceTest() throws JposException
    {

        fPtr.open(ui.getLogicalName());
        ui.displayText("Open(\"" + ui.getLogicalName() + "\")");
        try
        {
            // Common Properties
            ui.displayText("DeviceControlVersion = "
                    + fPtr.getDeviceControlVersion());
            ui.displayText("DeviceControlDescription = "
                    + fPtr.getDeviceControlDescription());
            ui.displayText("DeviceServiceVersion = "
                    + fPtr.getDeviceServiceVersion());
            ui.displayText("DeviceServiceDescription = "
                    + fPtr.getDeviceServiceDescription());
            ui.displayText("CapCompareFirmwareVersion = "
                    + fPtr.getCapCompareFirmwareVersion());
            String capPowerReportingName = "";
            switch (fPtr.getCapPowerReporting())
            {
                case JposConst.JPOS_PR_NONE:
                    capPowerReportingName += "JPOS_PR_NONE = ";
                    break;
                case JposConst.JPOS_PR_STANDARD:
                    capPowerReportingName += "JPOS_PR_STANDARD = ";
                    break;
                case JposConst.JPOS_PR_ADVANCED:
                    capPowerReportingName += "JPOS_PR_ADVANCED = ";
                    break;
            }
            ui.displayText("CapPowerReporting = " + capPowerReportingName
                    + fPtr.getCapPowerReporting());
            ui.displayText("CapStatisticsReporting = "
                    + fPtr.getCapStatisticsReporting());
            ui.displayText("CapUpdateFirmware = "
                            + fPtr.getCapUpdateFirmware());
            ui.displayText("CapUpdateStatistics = "
                    + fPtr.getCapUpdateStatistics());
            fPtr.addStatusUpdateListener(this);
            ui.displayText("addStatusUpdateListener(this)");
            fPtr.addDirectIOListener(this);
            ui.displayText("addDirectIOListener(this)");

            if (fPtr.getCapPowerReporting() != JposConst.JPOS_PR_NONE)
            {
                fPtr.setPowerNotify(JposConst.JPOS_PN_ENABLED);
                ui.displayText("PowerNotify = JPOS_PN_ENABLED");
            }

            fPtr.claim(1000);
            ui.displayText("Claim(1000)");

            synchronized (iplEndReceived)
            {
                fPtr.setDeviceEnabled(true);
                ui.displayText("setDeviceEnabled(true)");
                try
                {
                    iplEndReceived.wait(60000);
                }
                catch (InterruptedException e)
                {
                    ui.displayText("Timeout waiting IPL End");
                }
            }

            // Turn on notify
            int[] cmd = { DirectIO.FISCAL_NOTIFY };
            Integer fiscalNotify = new Integer(DirectIO.FISCAL_NOTIFY_ON);
            fPtr.directIO(DirectIO.FISCAL_ID, cmd, fiscalNotify);
            ui.displayText("FISCAL NOTIFY ON");

            ui.displayText("Get country cmd");
            byte[] cmdGetCountry = { (byte)0xFC, 0x00 };
            write(cmdGetCountry);

            ui.displayText("Customer Receipt station print test");
            byte[] cmdCR_Test = { (byte)0xE7, 0x08 };
            write(cmdCR_Test);

            ui.displayText("#\n# Open/close cover/s and/or press buttons " +
                           "to get status/directIO events...\n#");

            try
            {
                Thread.sleep(15000);
            }
            catch (InterruptedException e)
            {
            }

        }
        finally
        {
            fPtr.removeStatusUpdateListener(this);
            ui.displayText("removeStatusUpdateListener(this)");
            fPtr.removeDirectIOListener(this);
            ui.displayText("removeStatusDirectIOListener(this)");
            fPtr.close();
            ui.displayText("close()");
        }
    }

    /**
     * Read the data comming from the printer.
     * 
     * @param length
     *              the length of the data expected
     */
    protected void read(int length)
    {

        // subcommand
        int[] cmd =
        { DirectIO.FISCAL_READ };
        // return buffer
        byte[] buf = new byte[length];
        // read data
        try
        {
            fPtr.directIO(DirectIO.FISCAL_ID, cmd, buf);
        }
        catch (JposException e)
        {
            System.out.println("(READ)JposException with message = "
                    + e.getMessage() + " and errorCode = " + e.getErrorCode());
        }
    }

    /**
     * Sends a cmd to the printer.
     * 
     * @param buffer 
     *              Data to be sent to the printer
     */
    protected void write(byte[] buf)
    {

        // execute the FISCAL_WRITE subcommand. Send data to the Fiscal Printer.
        int[] cmd =
        { DirectIO.FISCAL_WRITE };
        try
        {
            synchronized (statusReceived)
            {
                try
                {
                    fPtr.directIO(DirectIO.FISCAL_ID, cmd, buf);
                    statusReceived.wait(30000);
                }
                catch (InterruptedException e)
                {
                    ui.displayText("Timeout waiting status received");
                }
            }
        }
        catch (JposException e)
        {
            System.out.println("(WRITE)JposException with message = "
                    + e.getMessage() + " and errorCode = " + e.getErrorCode());
        }
    }

    /**
     * Converts a byte[] into its hexadecimal representation string
     * 
     * @param cmdBuffer
     *            the buffer to convert
     * @return The hexadecimal string representation of this buffer
     */
    protected String byteArrayToHexString(byte[] cmdBuffer)
    {
        if (cmdBuffer == null)
            return "";

        String txt = new String("Hex:");
        String nbr;

        for (int i = 0; i < cmdBuffer.length; i++)
        {
            nbr = Integer.toHexString(cmdBuffer[i] & 0xFF);
            if (cmdBuffer[i] <= 0x0F)
            {
                nbr = "0" + nbr;
            }
            txt = txt + nbr + "-";
        }
        return txt;
    }

    // -------------------------------------------------------------------------
    // instance variables
    //
    private FiscalPrinter fPtr = new FiscalPrinter();

    private static DemoUI ui;

    private Object iplEndReceived = new Object();

    private Object statusReceived = new Object();

    // -------------------------------------------------------------------------
    // constants
    //
    /** Label for Start Button */
    private static final String TEXT_FPTR_TEST = "Start Fiscal Printer Test";

    /** tool tip text for Start button */
    private static final String TOOLTIP_TEXT_FPTR_TEST = 
                "- Open, claim and enable the FiscalPrinter " +
                "- Wait for IPL End " +
                "- Execute Get Country Code Command " +
                "- Execute CR Station Print Test " +
                "- Wait for Device statuses " +
                "- Close Fiscal Printer";
}
